//----------------------------------------------------------------------------
//
// Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
//
// Use, modification and distribution are subject to the Boost Software
// License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
//
//----------------------------------------------------------------------------

// This is an example program for using the COM interface of SIMCA-Q dll. To build and 
// run this application you must copy SIMCAQ.tlb to the same directory where you have 
// put the source files.
//
// You must also register the SIMCA-Q.dll with regsvr32.

#include "stdafx.h"
#include "SQPCOMSample.h"
#include "SQPCOMSampleDlg.h"

/////////////////////////////////////////////////////////////////////////////
// CAboutDlg dialog used for App About

class CAboutDlg : public CDialog
{
public:
   CAboutDlg() : CDialog(CAboutDlg::IDD) {}

   enum { IDD = IDD_ABOUTBOX };
};

/////////////////////////////////////////////////////////////////////////////
// CSQPCOMSampleDlg dialog

CSQPCOMSampleDlg::CSQPCOMSampleDlg(CWnd* pParent /*=NULL*/)
   : CDialog(CSQPCOMSampleDlg::IDD, pParent)
   , m_hIcon(AfxGetApp()->LoadIcon(IDR_MAINFRAME))
{
}

CSQPCOMSampleDlg::~CSQPCOMSampleDlg()
{
}

void CSQPCOMSampleDlg::DoDataExchange(CDataExchange* pDX)
{
   CDialog::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(CSQPCOMSampleDlg, CDialog)
   ON_WM_SYSCOMMAND()
   ON_WM_PAINT()
   ON_WM_QUERYDRAGICON()
   ON_BN_CLICKED(IDC_BUTTON1, OnStartSQP)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSQPCOMSampleDlg message handlers

BOOL CSQPCOMSampleDlg::OnInitDialog()
{
   CDialog::OnInitDialog();

   // Add "About..." menu item to system menu.

   // IDM_ABOUTBOX must be in the system command range.
   ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
   ASSERT(IDM_ABOUTBOX < 0xF000);

   CMenu* pSysMenu = GetSystemMenu(FALSE);
   if (pSysMenu != nullptr)
   {
      CString strAboutMenu;
      strAboutMenu.LoadString(IDS_ABOUTBOX);
      if (!strAboutMenu.IsEmpty())
      {
         pSysMenu->AppendMenu(MF_SEPARATOR);
         pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
      }
   }

   // Set the icon for this dialog.  The framework does this automatically
   //  when the application's main window is not a dialog
   SetIcon(m_hIcon, TRUE);			// Set big icon
   SetIcon(m_hIcon, FALSE);		// Set small icon

   // TODO: Add extra initialization here

   return TRUE;  // return TRUE  unless you set the focus to a control
}

void CSQPCOMSampleDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
   if ((nID & 0xFFF0) == IDM_ABOUTBOX)
   {
      CAboutDlg dlgAbout;
      dlgAbout.DoModal();
   }
   else
   {
      CDialog::OnSysCommand(nID, lParam);
   }
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CSQPCOMSampleDlg::OnPaint()
{
   if (IsIconic())
   {
      CPaintDC dc(this); // device context for painting

      SendMessage(WM_ICONERASEBKGND, (WPARAM)dc.GetSafeHdc(), 0);

      // Center icon in client rectangle
      int cxIcon = GetSystemMetrics(SM_CXICON);
      int cyIcon = GetSystemMetrics(SM_CYICON);
      CRect rect;
      GetClientRect(&rect);
      int x = (rect.Width() - cxIcon + 1) / 2;
      int y = (rect.Height() - cyIcon + 1) / 2;

      // Draw the icon
      dc.DrawIcon(x, y, m_hIcon);
   }
   else
   {
      CDialog::OnPaint();
   }
}

// The system calls this to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CSQPCOMSampleDlg::OnQueryDragIcon()
{
   return (HCURSOR)m_hIcon;
}

void CSQPCOMSampleDlg::OnStartSQP()
{
   CreateSQP();
}

void CSQPCOMSampleDlg::CreateSQP()
{
   CString strErr;
   CString strOutputFile;
   IPreparePredictionPtr pPrepPred;          // Will contain the data to prepare the prediction
   IPredictionPtr     pPrediction;  // Will contain the handle to the predictions.
   IModelPtr          pModel;   // Will contain the handle to a model.
   IVariableVectorPtr pVarVec;
   IVariablePtr       pVariable;

   wchar_t szTempPath[_MAX_DIR];
   GetTempPathW(_MAX_DIR, szTempPath);
   strOutputFile = szTempPath + CString(L"SIMCA-Q COM result.txt");

   try
   {
      int      iNumObs = 10;            // Number of observations to predict.
      long     lNumModels = 0;          // Number of models in the project.

      _bstr_t strProjectName;           // Will contain the project name
      _bstr_t strVarName;               // Will contain the name of a qualitative variable
      _bstr_t strVarNameSettings;       // Will contain the name of a qualitative variable

      IStringVectorPtr pstrVecXVarNames;              // Names of the x variables in a model
      IStringVectorPtr pstrVecQualNames;              // Names of the qualitative variables in a model
      IStringVectorPtr pstrVecNamesSettings;          // Holds the settings for a given qualitative variable.
      IIntVectorPtr    pnVecLagSteps;                 // vector that will contain how many steps to lag.

      CLSID clsSQ = {0};
      if ((CLSIDFromProgID(L"Umetrics.SIMCAQ", &clsSQ)) != S_OK)
         return;

      const auto hr = mpSQ.CreateInstance(clsSQ);
      if (hr != S_OK)
         throw("CreateInstance failed");

      // mpFile is a temp file where we store the results from our predictions
      mpFile = _wfopen(strOutputFile, L"w, ccs=UTF-8");

      if (mpFile == nullptr)
         throw("Could not open the result file.");

      CFileDialog oDlg(true, nullptr, nullptr,
         OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,
         L"SIMCA project (*.usp)|*.usp|All files (*.*)|*.*||");

      oDlg.m_ofn.lpstrTitle = L"Open SIMCA Project";

      if (IDOK == oDlg.DoModal())
      {
         CWaitCursor oWait;

         // Now an *.usp file has been selected.
         CString strUspPath = oDlg.GetPathName();

         fwprintf(mpFile, L"Path to usp file:        %s\n", strUspPath.GetString());

         mpProject = mpSQ->OpenProject(_bstr_t(strUspPath), _bstr_t(""));

         // Add the line below if you want to support python plugins
         // mpProject->EnableEmbeddedScripts();

         ///////////////////////////////////////////////
         // Get the project name
         //
         strProjectName = mpProject->GetProjectName();
         fwprintf(mpFile, L"Project name:            %s\n", static_cast<const wchar_t*>(strProjectName));

         /////////////////////////////////////////////////
         // Get number of models in the project.
         //
         lNumModels = mpProject->GetNumberOfModels();
         fwprintf(mpFile, L"Number of fitted models: %ld\n\n", lNumModels);

         /////////////////////////////////////////////////
         // Get info for each one of the fitted models.
         //
         for (long lModelIndex = 1; lModelIndex <= lNumModels; ++lModelIndex)
         {
            /////////////////////////////////////////////////
            // Get the model number connected with this index.
            //
            const auto nModelNumber = mpProject->GetModelNumberFromIndex(lModelIndex);

            pModel = mpProject->GetModel(nModelNumber);

            if (!pModel->IsModelFitted())
            {
               // The model is not fitted, go to the next model.
               fwprintf(mpFile, L"Model is not fitted.\n");
            }
            else
            {
               /////////////////////////////////////////////////
               // Get the name of the model connected with this index.
               //
               // Note: Model number is input NOT model index from now on.
               //
               _bstr_t strModelName = pModel->GetModelName();
               fwprintf(mpFile, L"==================================\n");
               fwprintf(mpFile, L"Model name: %s\n", static_cast<const wchar_t*>(strModelName));

               /////////////////////////////////////////////////
               // Create the prediction data
               pPrepPred = pModel->PreparePrediction();
               pVarVec = pPrepPred->GetVariablesForPrediction();
               int nVariables = pVarVec->GetSize();
               float fVal = 0;
               for (int iVar = 1; iVar <= nVariables; ++iVar)
               {
                  pVariable = pVarVec->GetVariable(iVar);
                  strVarName = pVariable->GetName(1);
                  BOOL bIsLagged = pVariable->IsLagged();
                  BOOL bIsQualiative = pVariable->IsQualitative();

                  // Get qualitative information
                  if (bIsQualiative)
                  {
                     pstrVecNamesSettings = pVariable->GetQualitativeSettings();
                     strVarNameSettings = pstrVecNamesSettings->GetData(1); // Just take the first one
                  }

                  // Set lag information
                  if (bIsLagged)
                  {
                     IIntVectorPtr pLagSteps;
                     int iNumLags;
                     int iMaxLag = 0;
                     pLagSteps = pVariable->GetLagSteps();
                     iNumLags = pLagSteps->GetSize();

                     for (int iLag = 1; iLag <= iNumLags; ++iLag)
                     {
                        int iLagStep;
                        iLagStep = pLagSteps->GetData(iLag);
                        iMaxLag = max(iMaxLag, iLagStep);

                        // Print the lagged variable
                        fwprintf(mpFile, L"%s.L%d\t", static_cast<const wchar_t*>(strVarName), iLagStep);
                     }
                     for (int iLag = 1; iLag <= iMaxLag; ++iLag)
                     {
                        if (bIsQualiative)
                           pPrepPred->SetQualitativeLagData(pVariable, iLag, strVarNameSettings); // Just take the first value
                        else
                           pPrepPred->SetQuantitativeLagData(pVariable, iLag, fVal++); // Just set to a fake value
                     }
                  }
                  else // Set non lagged prediction data
                  {
                     // Print the name of the variable
                     if (bIsQualiative)
                        fwprintf(mpFile, L"Qualitative variable: %s\n", static_cast<const wchar_t*>(strVarName));
                     else
                        fwprintf(mpFile, L"Quantitative variable: %s\n", static_cast<const wchar_t*>(strVarName));

                     for (int iRow = 1; iRow <= iNumObs; ++iRow)
                     {
                        if (bIsQualiative)
                           pPrepPred->SetQualitativeData(iRow, iVar, strVarNameSettings); // Just take the first value
                        else
                           pPrepPred->SetQuantitativeData(iRow, iVar, fVal++); // Just set to a fake value
                     }
                  }


               }

               /////////////////////////////////////////////////
               // Make the prediction
               pPrediction = pPrepPred->GetPrediction();

               fwprintf(mpFile, L"\n");
               // Get the "static" model data
               GetModelParameters(nModelNumber);
               fwprintf(mpFile, L"----------------------------------\n");
               fwprintf(mpFile, L"Predicted Result: \n");

               // Get the predicted result
               GetResults(pPrediction, nModelNumber);
            }
         }
      }
   }
   catch (const char* szError)
   {
      strErr = szError;
   }
   catch (const _com_error& err)
   {
      strErr.Append(mpSQ->GetErrorDescription(err.Error()));
   }
   catch (...)
   {
      strErr = L"Unknown error";
   }

   if (mpFile)
   {
      fclose(mpFile);
      mpFile = nullptr;
   }

   if (!strErr.IsEmpty())
   {
      AfxMessageBox(strErr);
   }
   else
   {
      if (_waccess(strOutputFile, 00) == 0)
      {
         ::ShellExecuteW(nullptr, L"open", strOutputFile, nullptr, nullptr, SW_SHOWNORMAL);
      }
   }
}

void CSQPCOMSampleDlg::GetModelParameters(long nModelNumber)
{
   long iNumComp = 0;
   BOOL bIsPLSModel;
   IModelPtr pModel;
   IVectorDataPtr pfVectorData;
   IFloatMatrixPtr pfMatrix;

   fwprintf(mpFile, L"----------------------------------\n");
   fwprintf(mpFile, L"Model data for model number %ld\n", nModelNumber);

   pModel = mpProject->GetModel(nModelNumber);

   bIsPLSModel = pModel->IsModelPLS();

   iNumComp = pModel->GetNumberOfComponents();
   fwprintf(mpFile, L"Number of components is %ld.\n", iNumComp);

   pfVectorData = pModel->GetContributionsScoresSingleWeight(0 /*iObs1Ix*/, 1/*iObs2Ix*/, eWeight_Normalized, iNumComp, 1, eReconstruct_False /*bReconstruct*/);
   pfMatrix = pfVectorData->GetDataMatrix();
   fwprintf(mpFile, L"Contribution SSW:\n");
   PrintFloatMatrix(pfMatrix);

   // GetModelP() is not valid for a zero component model. If it's called for with a model
   // that has zero components it will return false!
   if (iNumComp > 0)
   {
      pfVectorData = pModel->GetP(NULL /*pComponents*/, eReconstruct_False /*bReconstruct*/);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"P:\n");
      PrintFloatMatrix(pfMatrix);

      // GetModelT() is not valid for a zero component model. If it's called for with a model
      // that has no components it will return false!
      pfVectorData = pModel->GetT(NULL /*pComponents*/);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"T:\n");
      PrintFloatMatrix(pfMatrix);
   }

   /* Get T2Range from the first to the last component. */
   pfVectorData = pModel->GetT2Range(1, -1);
   pfMatrix = pfVectorData->GetDataMatrix();
   fwprintf(mpFile, L"T2Range:\n");
   PrintFloatMatrix(pfMatrix);

   if (bIsPLSModel == TRUE)
   {
      // C is only valid for a PLS model of some kind.

      // Get C for all components in the model.
      pfVectorData = pModel->GetC(NULL /*pComponents*/);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"C:\n");
      PrintFloatMatrix(pfMatrix);
   }
   else
   {
      /* Q2VX is only valid for a PCA model. */

      /* Get Q2VX */
      pfVectorData = pModel->GetQ2VX(NULL);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"Q2VX:\n");
      PrintFloatMatrix(pfMatrix);
   }
}

void CSQPCOMSampleDlg::GetResults(IPredictionPtr& pPrediction, long nModelNumber)
{
   IModelPtr pModel = mpProject->GetModel(nModelNumber);
   IVectorDataPtr pfVectorData;
   IFloatMatrixPtr pfMatrix;

   /////////////////////////////////////////////////
   // Get the number of components for this model
   //
   long lNumComp = pModel->GetNumberOfComponents();
   fwprintf(mpFile, L"-------------\n");

   if (lNumComp > 0)
   {
      /////////////////////////////////////////////////
      // Get contribution for the prediction
      //
      // We will request a Single weight score contribution with no weight and
      // the average as the reference. This particular request can only be made for
      // a model that has one or more components.
      //
      Weight eWeight = eWeight_Normalized;
      long lObs1Ix = 0;        // Use the average as reference.
      long lObs2Ix = 1;        // We want contribution for the first observation.
      long lYVar = 1;       // Whatever, this par. will be ignored since the weight is RX.
      long lComponent = lNumComp; // For this sample. Set to the number of components in the model.

      pfVectorData = pPrediction->GetContributionsScorePSSingleWeight(lObs1Ix, lObs2Ix, eWeight, lComponent, lYVar, eReconstruct_False);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"Predicted Score Contribution:\n");
      PrintFloatMatrix(pfMatrix);

      /////////////////////////////////////////////////
      // Get the predicted T
      //
      // GetPredictedT() is not valid for a zero component model. If it's called for with a model
      // that has no components it will return false!

      pfVectorData = pPrediction->GetTPS(NULL /*pComponentList*/);
      pfMatrix = pfVectorData->GetDataMatrix();
      fwprintf(mpFile, L"TPS:\n");
      PrintFloatMatrix(pfMatrix);
   }

   /////////////////////////////////////////////////
   // Get the predicted DModX.
   //
   pfVectorData = pPrediction->GetDModXPS(NULL /*pComponentList*/, eNormalized_True, eModelingPowerWeighted_False);
   pfMatrix = pfVectorData->GetDataMatrix();
   fwprintf(mpFile, L"DModXPS:\n");
   PrintFloatMatrix(pfMatrix);

   /////////////////////////////////////////////////
   // Get the predicted T2
   //
   // GetPredictedT2Range() is not valid for a zero component model. If it's called with a model
   // that has no components it will return false!
   /* Get T2RangePS */
   pfVectorData = pPrediction->GetT2RangePS(1, -1);
   pfMatrix = pfVectorData->GetDataMatrix();
   fwprintf(mpFile, L"T2RangePS:\n");
   PrintFloatMatrix(pfMatrix);

   /////////////////////////////////////////////////
   // Get the predicted XVar.
   //
   pfVectorData = pPrediction->GetXVarPS(eUnscaled_True, eBacktransformed_True, NULL);
   pfMatrix = pfVectorData->GetDataMatrix();
   fwprintf(mpFile, L"XVarPS:\n");
   PrintFloatMatrix(pfMatrix);
}

void CSQPCOMSampleDlg::PrintFloatMatrix(IFloatMatrixPtr& pfMat)
{
   float fVal;
   for (int iColIter = 1; iColIter <= pfMat->GetNumberOfCols(); iColIter++)
   {
      for (int iRowIter = 1; iRowIter <= pfMat->GetNumberOfRows(); iRowIter++)
      {
         fVal = pfMat->GetData(iRowIter, iColIter);
         fwprintf(mpFile, L"%f\t", fVal);
      }
      fwprintf(mpFile, L"\n");
   }
   fwprintf(mpFile, L"\n");
}
